/******************************************************************************\

Module Name:    LAudioIn.c

Description:    LAudioIn sample implementation file.

References:     LAudioIn.doc.
                Matrox Liberatus specifications

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------
#ifdef UNIX
#include <termios.h>
#include <unistd.h>
#endif

#include "Liberatus.h"
#include "LBuffer.h"
#include "LAudioIn.h"

#include "stdio.h"
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define ABORT_TRY
#define ABORT_CATCH             Abort:
#define ABORT_NOW()             goto Abort;
#define ABORT_IF(cond)          if(cond) { ABORT_NOW(); }
#define ABORT_ON_LERR(eStatus)  ABORT_IF(LSTATUS_IS_FAIL(eStatus))

typedef struct
{
    const char* szOutputFilePath;
    MUINT32     uiInputIdx;
    MBOOL32     bPrintInfo;
    MBOOL32     bPrintEvents;
    MUINT32     uiSamplesPerBuf;

} MainArgs;

// --------------------------------------------------------------------------------------
//                                         C O D E
// --------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ParseArgs

Description:    Parse application arguments.

Comments:       None.

\************************************************************************************************************/
void ParseArgs(int argc, char** argv, MainArgs* poArgs, MUINT32 uiAinCount)
{
    MBOOL32 bInvalidArg = MFALSE;
    MUINT i = 0;

    poArgs->szOutputFilePath    = MNULL;
    poArgs->uiInputIdx          = 0;
    poArgs->bPrintInfo          = MFALSE;
    poArgs->bPrintEvents        = MFALSE;
    poArgs->uiSamplesPerBuf     = 1024;

    for (i = 1; i < argc; i++)
    {
        if ((strcmp(argv[i], "-o") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->szOutputFilePath = argv[i];
        }
        else if ((strcmp(argv[i], "-input") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiInputIdx = strtoul(argv[i], MNULL, 10);

            if (poArgs->uiInputIdx > uiAinCount)
            {
                bInvalidArg = MTRUE;
            }
        }
        else if ((strcmp(argv[i], "-spb") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiSamplesPerBuf = strtoul(argv[i], MNULL, 10);
        }
        else if (strcmp(argv[i], "-print-info") == 0)
        {
            poArgs->bPrintInfo = MTRUE;
        }
        else if (strcmp(argv[i], "-print-events") == 0)
        {
            poArgs->bPrintEvents = MTRUE;
        }
        else
        {
            bInvalidArg = MTRUE;
        }

        if (bInvalidArg)
        {
            break;
        }
    }

    if (bInvalidArg)
    {
        printf("Usage: %s [-o output_file]\n", argv[0]);
        printf("Optional arguments:\n");
        printf("\t-o str        : Output file path\n");
        printf("\t-input n      : Audio input index (0 to %u, default=0)\n", uiAinCount - 1);
        printf("\t-spb n        : Samples per buffer (default=1024).\n");
        printf("\t-print-info   : Print captured buffers informations.\n");
        printf("\t-print-events : Print received events\n");
        exit(1);
    }
}

/************************************************************************************************************\

Function:       GetChar

Description:    Non blocking getch.

Comments:       None.

\************************************************************************************************************/
int GetChar()
{
#ifdef _WIN32
    int ch = _getch();

    /*In linux, a keypress enter is a LFeed with a value of 10. In Windows,
    The enter key calls a carriage return which has an ASCII value of 13 instead
    To avoid having to modify it at each call of getchar we go it once here so that
    it's consistant in what it expects between Windows and Linux*/
    if (ch == 13)
        ch = 10;
    return ch;
#else
    struct termios oOldAttrib;
    struct termios oNewAttrib;

    tcgetattr(STDIN_FILENO, &oOldAttrib);
    oNewAttrib = oOldAttrib;
    oNewAttrib.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &oNewAttrib);

    int iOldFlags = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags | O_NONBLOCK);

    int ch = getchar();

    tcsetattr(STDIN_FILENO, TCSANOW, &oOldAttrib);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags);

    return ch;
#endif

}

/************************************************************************************************************\

Function:       PrintDescriptor

Description:    Prints LAudioFormat_Descriptor.

Comments:       None.

\************************************************************************************************************/
void PrintDescriptor(LAudioFormat_Descriptor* poDescriptor)
{
    MUINT32 i = 0;

    printf("AudioFormat:        0x%08x\n"
           "Sampling rate:      %u\n"
           "Sample size:        %u\n"
           "Number of channels: %u\n"
           "Number of plans:    %u\n"
           "PCM:                %u\n",
           poDescriptor->eAudioFormat,
           poDescriptor->uiSampleRate,
           poDescriptor->uiSampleSize,
           poDescriptor->uiNumberOfChannels,
           poDescriptor->uiNumberOfPlans,
           poDescriptor->bPcmData);

    printf("Layout:");
    for(i=0; i < poDescriptor->uiNumberOfPlans; i++)
    {
        printf("             [%d]:[%08x] ", i, poDescriptor->aflPlansChannelLayout[i]);
    }
    printf("\n");
}

/************************************************************************************************************\

Function:       PrintFormat

Description:    Print LAudioFormat descriptor.

Comments:       None.

\************************************************************************************************************/
LStatus PrintFormat(LAudioFormat eAudioFormat)
{
    LAudioFormat_Descriptor oDescriptor = {LAudioFormat_INVALID};

    LStatus eStatus = LAudioFormat_GetDescriptor(eAudioFormat, &oDescriptor);

    if((eStatus == LStatus_OK)
       && (eAudioFormat == oDescriptor.eAudioFormat))
    {
        PrintDescriptor(&oDescriptor);
        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       HandleEvents

Description:    Get and print LAudioIn events.

Comments:       None.

\************************************************************************************************************/
void HandleEvents(LAudioIn_Handle hAudioIn)
{
    LStatus eStatus = LStatus_OK;
    LAudioIn_Event eEvent = LAudioIn_Event_INVALID;

    eStatus = LAudioIn_WaitForEvent(hAudioIn, 0, &eEvent);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        switch(eEvent)
        {
            case LAudioIn_Event_OUT_OF_BUFFER:
            {
                printf("Received OUT_OF_BUFFER event.\n");
                break;
            }
            case LAudioIn_Event_ALL_BUFFER_RELEASED:
            {
                printf("Received ALL_BUFFER_RELEASED event.\n");
                break;
            }
            case LAudioIn_Event_SOURCE_CHANGED:
            {
                printf("Received SOURCE_CHANGED event.\n");
                break;
            }
            case LAudioIn_Event_START_PENDING:
            {
                printf("Received START_PENDING event.\n");
                break;
            }
            case LAudioIn_Event_STARTED:
            {
                printf("Received STARTED event.\n");
                break;
            }
            case LAudioIn_Event_STATE_CHANGED:
            {
                printf("Received STATE_CHANGED event.\n");
                break;
            }
            case LAudioIn_Event_STOPPED:
            {
                printf("Received STOPPED event.\n");
                break;
            }
            case LAudioIn_Event_SOURCE_DETECTED:
            {
                printf("Received SOURCE_DETECTED event.\n");
                break;
            }
            default:
            {
                printf("Unknown event: %d.\n", eEvent);
                break;
            }
        }
    }
    else if (eStatus != LStatus_TIMEOUT)
    {
        printf("ERROR in LAudioIn_WaitForEvent: %d\n", eStatus);
    }
}

/********************************************************************\

Function:       PrintCaptureState

Description:    Print the current state of the capture.

Comments:       None.

\********************************************************************/
LStatus PrintCaptureState(LAudioIn_Handle hAudioIn)
{
    LStatus eStatus = LStatus_FAIL;

    LAudio_State    eState                  = LAudio_State_INVALID;
    MUINT32         uiTotalBufferCount      = 0;
    MUINT32         uiReadyBufferCount      = 0;
    MUINT32         uiLoanedBufferCount     = 0;
    MUINT32         uiInternalBufferCount   = 0;

    eStatus = LAudioIn_GetCaptureState(
                    hAudioIn,
                    &eState,
                    &uiTotalBufferCount,
                    &uiReadyBufferCount,
                    &uiLoanedBufferCount,
                    &uiInternalBufferCount);

    if(eStatus == LStatus_OK)
    {
        MUINT64 uiLastBufferCounter = 0;
        MUINT64 uiLastTickRefCounter = 0;

        eStatus = LAudioIn_GetLastBufferCounter(
                        hAudioIn,
                        &uiLastBufferCounter,
                        &uiLastTickRefCounter);

        if(eStatus == LStatus_OK)
        {
            printf("Cnt=%lu - Tc=%lu - Buf[Tot=%u, Rdy=%u, Lnd=%u]\n",
                    uiLastBufferCounter,
                    uiLastTickRefCounter,
                    uiTotalBufferCount,
                    uiReadyBufferCount,
                    uiLoanedBufferCount);
        }
    }

    return eStatus;
}

/********************************************************************\

Function:       main

Description:    LAudioIn sample entry point.

Comments:

\********************************************************************/
int main(int argc, char** argv)
{
    MINT                    iRet            = 1;
    LStatus                 eStatus         = LStatus_FAIL;
    LDevice_Handle          hDevice         = MNULL;
    LAudioIn_Handle         hAudioIn        = MNULL;
    LAudioFormat            eFormat         = LAudioFormat_INVALID;
    MUINT32                 uiBufferSize    = 0;
    LBuffer_Handle          hBuffer         = MNULL;
    MUINT8*                 puiOutBuffer    = MNULL;
    MUINT8*                 puiInBufAddress = MNULL;
    MainArgs                oArgs           = {0};
    FILE*                   hOutFile        = MNULL;
    MBOOL32                 bBuffersCreated = MFALSE;
    MBOOL32                 bStarted        = MFALSE;

    ABORT_TRY
    {
        Liberatus_Load();

        //============================================
        // Get Liberatus handle.
        hDevice = Liberatus_GetDevice(0);
        ABORT_IF(hDevice == MNULL);

        MUINT32 uiAinCount = 0;
        eStatus = LAudioIn_GetCount(hDevice, &uiAinCount);
        ABORT_ON_LERR(eStatus);

        if (uiAinCount == 0)
        {
            printf("No audio inputs on this device.\n");
            ABORT_NOW();
        }

        //============================================
        // Parse main arguments.
        ParseArgs(argc, argv, &oArgs, uiAinCount);

        //============================================
        // Get LAudioIn handle.
        eStatus = LAudioIn_GetHandle(hDevice,
                               oArgs.uiInputIdx,
                               LAccessMode_READWRITE_EXCLUSIVE,
                               &hAudioIn);
        ABORT_ON_LERR(eStatus);

        //============================================
        // Detect audio input signal.
        MBOOL32 bIsDetected = MFALSE;
        MBOOL32 bIsCapturable = MFALSE;

        eStatus = LAudioIn_DetectSource(hAudioIn, &bIsCapturable, &bIsDetected);

        if (eStatus != LStatus_UNSUPPORTED)
        {
            ABORT_ON_LERR(eStatus);

            printf("Audio input signal detected:%s, capturable:%s\n",
                   bIsDetected   ? "yes" : "no",
                   bIsCapturable ? "yes" : "no");

            LAudioFormat_Descriptor oDescriptor = {0};
            eStatus = LAudioIn_GetDetectedAudioParameters(hAudioIn, &oDescriptor);
            ABORT_ON_LERR(eStatus);

            printf("======== Input signal parameters ========\n");
            PrintDescriptor(&oDescriptor);
            printf("==========================================\n\n");
        }
        else
        {
            printf("Source detection not supported on this device.\n\n");
        }

        eFormat = LAudioFormat_INVALID;
        eStatus = LAudioIn_EnumSupportedAudioFormat(hAudioIn, 0, &eFormat);
        ABORT_ON_LERR(eStatus);

        printf("======= Captured stream parameters =======\n");
        eStatus = PrintFormat(eFormat);
        printf("==========================================\n\n");
        ABORT_ON_LERR(eStatus);

        //============================================
        // Create LAudioIn input buffers.
        uiBufferSize = 0;

        eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(eFormat, oArgs.uiSamplesPerBuf, &uiBufferSize);
        ABORT_ON_LERR(eStatus);

        printf("Creating buffers (4 * %u bytes)...\n", uiBufferSize);
        eStatus = LAudioIn_CreateBuffers(hAudioIn, eFormat, uiBufferSize, 4);
        ABORT_ON_LERR(eStatus);
        bBuffersCreated = MTRUE;

        LAudioFormat    eBufferFormat;
        MUINT32         uiNbBuffers;
        eStatus = LAudioIn_GetBufferAttributes(hAudioIn, &eBufferFormat, &uiBufferSize, &uiNbBuffers);
        ABORT_ON_LERR(eStatus);

        printf("=========== Buffer attributes ============\n");
        printf("Buffer size:        %u\n", uiBufferSize);
        printf("Number of buffers:  %u\n", uiNbBuffers);
        printf("==========================================\n\n");
        ABORT_ON_LERR(eStatus);

        //============================================
        // Create output buffer.
        eStatus = LStatus_OUT_OF_MEMORY;
        puiOutBuffer = (MUINT8*) malloc(uiBufferSize);
        ABORT_IF(puiOutBuffer == MNULL);

        //============================================
        // Start audio capture.
        printf("Starting capture...\n");
        eStatus = LAudioIn_StartCapture(hAudioIn);
        ABORT_ON_LERR(eStatus);
        bStarted = MTRUE;

        //============================================
        // Open output file.
        if (oArgs.szOutputFilePath != MNULL)
        {
            hOutFile = fopen(oArgs.szOutputFilePath, "wb+");
            ABORT_IF(hOutFile == MNULL);
        }

        MINT iChar = 0;

        printf("Capture started, hit 'q' to exit.\n");

        while (((eStatus == LStatus_OK) || (eStatus == LStatus_TIMEOUT))
               && (iChar != 'q'))
        {
            if (oArgs.bPrintEvents)
            {
                HandleEvents(hAudioIn);
            }

            eStatus = LAudioIn_GetNextBuffer(
                          hAudioIn,
                          MTRUE,
                          1000,
                          &hBuffer,
                          MNULL,
                          MNULL,
                          MNULL);

            ABORT_IF((eStatus != LStatus_OK) && (eStatus != LStatus_TIMEOUT));

            if (eStatus == LStatus_OK)
            {
                if (hOutFile != MNULL)
                {
                    eStatus = LBuffer_BeginAccess(
                                  hBuffer,
                                  0,
                                  1,
                                  &puiInBufAddress);

                    ABORT_ON_LERR(eStatus);

                    fwrite(puiInBufAddress, 1, uiBufferSize, hOutFile);

                    eStatus = LBuffer_EndAccess(hBuffer);
                    ABORT_ON_LERR(eStatus);

                    puiInBufAddress = MNULL;
                }

                eStatus = LAudioIn_ReleaseBuffer(hAudioIn, hBuffer);
                ABORT_ON_LERR(eStatus);
                hBuffer = MNULL;

                if (oArgs.bPrintInfo)
                {
                    eStatus = PrintCaptureState(hAudioIn);
                    ABORT_ON_LERR(eStatus);
                }
            }
            else
            {
                printf("GetNextBuffer timeout.\n");
            }

            iChar = GetChar();
        }

        iRet = 0;
    }
    ABORT_CATCH
    {
        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("Failed with status %d\n", eStatus);
        }

        //============================================
        // Cleanup.

        if (hBuffer != MNULL)
        {
            if (puiInBufAddress != MNULL)
            {
                LBuffer_EndAccess(hBuffer);
                puiInBufAddress = MNULL;
            }

            LAudioIn_ReleaseBuffer(hAudioIn, hBuffer);
        }

        if (hOutFile != MNULL)
        {
            fclose(hOutFile);
        }

        if (puiOutBuffer != MNULL)
        {
            free(puiOutBuffer);
            puiOutBuffer = MNULL;
        }

        if (bStarted)
        {
            eStatus = LAudioIn_StopCapture(hAudioIn);

            if (eStatus != LStatus_OK)
            {
                printf("ERROR in LAudioIn_StopCapture: %d\n", eStatus);
            }

            bStarted = MFALSE;
        }

        if (bBuffersCreated)
        {
            LAudioIn_DestroyBuffers(hAudioIn);
            bBuffersCreated = MFALSE;
        }

        if (hAudioIn != MNULL)
        {
            LAudioIn_ReleaseHandle(hAudioIn);
            hAudioIn = MNULL;
        }

        Liberatus_UnLoad();

        return iRet;
    }
}
